let provider = null;
let signer = null;
let userAddress = null;

// ===== CONFIG =====
const REQUIRED_CHAIN_ID = 97;      // BSC Testnet
const REQUIRED_CHAIN_HEX = "0x61";

// ===== UI =====
const connectBtn = document.getElementById("connectBtn");
const walletStatus = document.getElementById("walletStatus");
const networkStatus = document.getElementById("networkStatus");

/* ==========================
   UTILIDADES
========================== */

function isMetaMaskInstalled() {
  return typeof window.ethereum !== "undefined";
}

function shortAddress(addr) {
  return addr.slice(0, 6) + "..." + addr.slice(-4);
}

function resetUI() {
  walletStatus.innerText = "No conectada";
  walletStatus.className = "";
  networkStatus.innerText = "Desconocida";
  networkStatus.className = "";
}

/* ==========================
   RED
========================== */

function updateNetworkUI(chainId) {
  networkStatus.innerText = `Chain (${chainId})`;

  if (chainId === REQUIRED_CHAIN_ID) {
    networkStatus.className = "ok";
  } else {
    networkStatus.className = "blocked";
  }
}

async function requestNetworkChange() {
  try {
    await window.ethereum.request({
      method: "wallet_switchEthereumChain",
      params: [{ chainId: REQUIRED_CHAIN_HEX }]
    });
  } catch (error) {
    if (error.code === 4902) {
      await addBscTestnet();
    }
  }
}

async function addBscTestnet() {
  await window.ethereum.request({
    method: "wallet_addEthereumChain",
    params: [{
      chainId: REQUIRED_CHAIN_HEX,
      chainName: "BSC Testnet",
      nativeCurrency: {
        name: "tBNB",
        symbol: "tBNB",
        decimals: 18
      },
      rpcUrls: ["https://data-seed-prebsc-1-s1.binance.org:8545/"],
      blockExplorerUrls: ["https://testnet.bscscan.com"]
    }]
  });
}

/* ==========================
   WALLET
========================== */

async function connectWallet() {
  if (!isMetaMaskInstalled()) {
    alert("MetaMask no está instalado");
    return;
  }

  try {
    provider = new ethers.BrowserProvider(window.ethereum);

    // Solicitar cuentas
    await provider.send("eth_requestAccounts", []);

    // Comprobar red actual (usando ethereum directamente)
    const chainIdHex = await window.ethereum.request({
      method: "eth_chainId"
    });

    const chainId = parseInt(chainIdHex, 16);
    updateNetworkUI(chainId);

    if (chainId !== REQUIRED_CHAIN_ID) {
      await requestNetworkChange();
      return;
    }

    signer = await provider.getSigner();
    userAddress = await signer.getAddress();

    walletStatus.innerText = shortAddress(userAddress);
    walletStatus.className = "ok";

    connectBtn.innerText = "Desconectar";
    connectBtn.onclick = disconnectWallet;

  } catch (error) {
    console.error(error);
  }
}

function disconnectWallet() {
  provider = null;
  signer = null;
  userAddress = null;

  resetUI();

  connectBtn.innerText = "Conectar con MetaMask";
  connectBtn.onclick = connectWallet;
}

/* ==========================
   EVENTOS METAMASK
========================== */

if (window.ethereum) {

  // Cambio de cuenta
  window.ethereum.on("accountsChanged", (accounts) => {
    if (accounts.length === 0) {
      disconnectWallet();
    } else {
      disconnectWallet();
      connectWallet();
    }
  });

  // Cambio de red (CRÍTICO: recrear provider)
  window.ethereum.on("chainChanged", (chainIdHex) => {
    const chainId = parseInt(chainIdHex, 16);

    // Destruir estado anterior
    provider = null;
    signer = null;
    userAddress = null;

    // Crear provider limpio
    provider = new ethers.BrowserProvider(window.ethereum);

    updateNetworkUI(chainId);

    if (chainId !== REQUIRED_CHAIN_ID) {
      disconnectWallet();
    }
  });
}

/* ==========================
   INIT
========================== */

resetUI();
connectBtn.onclick = connectWallet;
