// ===== CONFIG =====
const CONTRACT_ADDRESS = "0x447cFE3003323B8E5B341542aE098c0953C6Afb9";

const ERC20_TOKEN_ADDRESS = "0x4e2afAc3e86b3ad045fF6E3e0aBBBEE58d33ED64";
const APPROVE_AMOUNT = "120";
const TOKEN_APPROVE_AMOUNT = "120";

const ERC20_ABI = [
  "function approve(address spender, uint256 amount) external returns (bool)",
  "function allowance(address owner, address spender) external view returns (uint256)",
  "function decimals() external view returns (uint8)",
  "function symbol() external view returns (string)"
];

// ===== UI =====
const addressInput = document.getElementById("addressInput");
const btnUserCount = document.getElementById("btnUserCount");
const btnUserData = document.getElementById("btnUserData");
const userCountResult = document.getElementById("userCountResult");
const userDataResult = document.getElementById("userDataResult");

// ===== UI (WRITE) =====
const sponsorInput = document.getElementById("sponsorInput");
const btnRegister = document.getElementById("btnRegister");
const registerResult = document.getElementById("registerResult");

const btnApprove = document.getElementById("btnApprove");
const approveResult = document.getElementById("approveResult");

const btnApproveRegister = document.getElementById("btnApproveRegister");
const approveRegisterResult = document.getElementById("approveRegisterResult");


/* ==========================
   HELPERS
========================== */

// function getContract(readonly = true) {
//   if (!window.ethereum || !provider) {
//     alert("Wallet no conectada");
//     return null;
//   }

//   return new ethers.Contract(
//     CONTRACT_ADDRESS,
//     CONTRACT_ABI,
//     readonly ? provider : signer
//   );
// }

function getContract(readonly = true) {
  if (!window.ethereum || !provider) {
    alert("Wallet no conectada");
    return null;
  }

  return new ethers.Contract(
    CONTRACT_ADDRESS,
    CONTRACT_ABI,
    readonly ? provider : signer
  );
}

function getErc20Contract() {
  if (!signer) {
    alert("Conecta la wallet primero");
    return null;
  }

  return new ethers.Contract(ERC20_TOKEN_ADDRESS, ERC20_ABI, signer);
}

function getContractWrite() {
  if (!signer) {
    alert("Conecta la wallet primero");
    return null;
  }
  return new ethers.Contract(CONTRACT_ADDRESS, CONTRACT_ABI, signer);
}


function isValidAddress(addr) {
  return ethers.isAddress(addr);
}

/* ==========================
   CONTRACT CALLS
========================== */

async function loadUserCount() {
  try {
    const contract = getContract(true);
    if (!contract) return;

    const count = await contract.userCount();
    userCountResult.innerHTML =
      `<span class="ok">userCount:</span> ${count.toString()}`;

  } catch (err) {
    console.error(err);
    userCountResult.innerHTML =
      `<span class="error">Error leyendo userCount</span>`;
  }
}

// async function loadUserData() {
//   try {
//     const addr = addressInput.value.trim();
    
//     if (!isValidAddress(addr)) {
//       alert("Address inválida");
//       return;
//     }
    
//     const contract = getContract(true);
//     if (!contract) return;
    
//     const user = await contract.users(addr);
    
//     console.log('Datos cargados:', user);
//     // userResultEl.textContent = user.toString();
//     // userDataResult.textContent = JSON.stringify(user, null, 2);

//     // Si es struct → mostrar como JSON
//     if (typeof user === "object") {
//         userResultEl.textContent = JSON.stringify(user, replacer, 2);
//     } else {
//         userResultEl.textContent = user.toString();
//     }

//   } catch (err) {
//     console.error(err);
//     userDataResult.textContent = "Error leyendo users(address)";
//   }
// }

async function loadUserData() {
  try {
    const addr = addressInput.value.trim();

    if (!ethers.isAddress(addr)) {
      alert("Address inválida");
      return;
    }

    const contract = getContract(true);
    if (!contract) return;

    const user = await contract.users(addr);
    console.log(user.referrals)

    // Normalizar struct (BigInt -> string)
    const normalizedUser = {
      id: user.id.toString(),
      sponsor: user.sponsor,
      createdAt: user.createdAt.toString(),
      activatedAt: user.activatedAt.toString(),
      totalReceived: user.totalReceived.toString(),
      totalLost: user.totalLost.toString(),
      isActive: user.isActive,
      canUpdateRank: user.canUpdateRank,
      rank: Number(user.rank),
      rankAssignedAt: user.rankAssignedAt.toString(),
      referrals: user.referrals
    };

    userDataResult.textContent = JSON.stringify(normalizedUser, null, 2);

  } catch (err) {
    console.error(err);
    userDataResult.textContent = "Error leyendo users(address)";
  }
}

async function registerWithPayment() {
  try {
    if (!signer) {
      alert("Debes conectar la wallet primero");
      return;
    }

    const sponsor = sponsorInput.value.trim();

    if (!ethers.isAddress(sponsor)) {
      alert("Sponsor address inválida");
      return;
    }

    const contract = getContract(false); // ⚠️ WRITE
    if (!contract) return;

    registerResult.textContent = "⏳ Enviando transacción...";

    // Enviar TX
    const tx = await contract.registerWithPayment(sponsor);

    registerResult.textContent =
      `📤 Transacción enviada\nHash: ${tx.hash}`;

    // Esperar confirmación
    const receipt = await tx.wait();

    registerResult.textContent =
      `✅ Transacción confirmada\nHash: ${tx.hash}\nBloque: ${receipt.blockNumber}`;

  } catch (err) {
    console.error(err);

    // Errores comunes de MetaMask
    if (err.code === 4001) {
      registerResult.textContent = "❌ Transacción rechazada por el usuario";
    } else {
      registerResult.textContent = "❌ Error al ejecutar registerWithPayment";
    }
  }
}

async function approveTokens() {
  try {
    const token = getErc20Contract();
    if (!token) return;

    approveResult.textContent = "⏳ Preparando aprobación...";

    // Leer decimales reales del token
    const decimals = await token.decimals();
    const symbol = await token.symbol();

    const amount = ethers.parseUnits(APPROVE_AMOUNT, decimals);

    // Comprobar allowance actual
    const currentAllowance = await token.allowance(
      await signer.getAddress(),
      CONTRACT_ADDRESS
    );

    if (currentAllowance >= amount) {
      approveResult.textContent =
        `✅ Ya existe allowance suficiente (${APPROVE_AMOUNT} ${symbol})`;
      return;
    }

    approveResult.textContent = "📤 Enviando aprobación...";

    const tx = await token.approve(CONTRACT_ADDRESS, amount);

    approveResult.textContent =
      `📤 Aprobación enviada\nHash: ${tx.hash}`;

    const receipt = await tx.wait();

    approveResult.textContent =
      `✅ Aprobación confirmada\n${APPROVE_AMOUNT} ${symbol}\nBloque: ${receipt.blockNumber}`;

  } catch (err) {
    console.error(err);

    if (err.code === 4001) {
      approveResult.textContent = "❌ Aprobación rechazada por el usuario";
    } else {
      approveResult.textContent = "❌ Error al aprobar tokens";
    }
  }
}



/* ==========================
   FLOW: Approve + Register
========================== */

async function approveAndRegister() {
  try {
    const sponsor = sponsorInput.value.trim();
    if (!isValidAddress(sponsor)) {
      alert("Sponsor address inválida");
      return;
    }

    if (!signer) {
      alert("Wallet no conectada");
      return;
    }

    approveRegisterResult.textContent = "⏳ Preparando proceso...";

    // ---- 1️⃣ Revisar allowance ----
    const token = getErc20Contract();
    const decimals = await token.decimals();
    const symbol = await token.symbol(); 
    const amount = ethers.parseUnits(TOKEN_APPROVE_AMOUNT, decimals);

    const ownerAddress = await signer.getAddress();
    const currentAllowance = await token.allowance(ownerAddress, CONTRACT_ADDRESS);

    if (currentAllowance < amount) {
      approveRegisterResult.textContent = `📤 Allowance insuficiente, aprobando ${TOKEN_APPROVE_AMOUNT} ${symbol}...`;

      const approveTx = await token.approve(CONTRACT_ADDRESS, amount);
      approveRegisterResult.textContent = `⏳ Aprobar TX enviada\nHash: ${approveTx.hash}`;
      await approveTx.wait();
      approveRegisterResult.textContent = `✅ Aprobar TX confirmada\n${TOKEN_APPROVE_AMOUNT} ${symbol} aprobados`;
    } else {
      approveRegisterResult.textContent = `✅ Allowance suficiente (${TOKEN_APPROVE_AMOUNT} ${symbol})`;
    }

    // ---- 2️⃣ Ejecutar registerWithPayment ----
    const contract = getContractWrite();
    approveRegisterResult.textContent += "\n📤 Enviando registerWithPayment...";

    const tx = await contract.registerWithPayment(sponsor);
    approveRegisterResult.textContent += `\n⏳ TX enviada\nHash: ${tx.hash}`;

    const receipt = await tx.wait();
    approveRegisterResult.textContent += `\n✅ Registro confirmado\nBloque: ${receipt.blockNumber}`;

  } catch (err) {
    console.error(err);
    if (err.code === 4001) {
      approveRegisterResult.textContent = "❌ Transacción rechazada por el usuario";
    } else {
      approveRegisterResult.textContent = "❌ Error en approve + register";
    }
  }
}


/* ==========================
   EVENTS
========================== */

btnUserCount.onclick = loadUserCount;
btnUserData.onclick = loadUserData;
btnRegister.onclick = registerWithPayment;

btnApprove.onclick = approveTokens;
btnApproveRegister.onclick = approveAndRegister;


// Autollenar input cuando conectas MetaMask
if (window.ethereum) {
  window.ethereum.on("accountsChanged", (accounts) => {
    if (accounts.length > 0) {
      addressInput.value = accounts[0];
    }
  });
}

// Autollenar input con MetaMask
if (window.ethereum) {
  window.ethereum.on("accountsChanged", (accounts) => {
    if (accounts.length > 0) {
      sponsorInput.value = accounts[0];
    }
  });
}